﻿using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.GUI;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;

namespace WPF.PresentationLayer.UC115.ViewModels
{
    public abstract class PendingTestViewModelBase : ViewModelBase
    {
        /// <summary>
        /// Command to add a review comment
        /// </summary>
        public DelegateCommand<ISelectableTestViewModel> CommentCommand { get; private set; }
        /// <summary>
        /// Command to close the window
        /// </summary>
        public DelegateCommand CloseCommand { get; set; }

        private bool enableRejectButton;
        /// <summary>
        /// EnableRejectButton
        /// </summary>
        public bool EnableRejectButton
        {
            get
            {
                return enableRejectButton;
            }
            set
            {
                enableRejectButton = value;
                RaisePropertyChanged(() => EnableRejectButton);
            }
        }

        private bool enableAcceptButton;
        /// <summary>
        /// EnableAcceptButton
        /// </summary>
        public bool EnableAcceptButton
        {
            get
            {
                return enableAcceptButton;
            }
            set
            {
                enableAcceptButton = value;
                RaisePropertyChanged(() => EnableAcceptButton);
            }
        }
        /// <summary>
        /// Help File Topic 
        /// </summary>
        public StrRes.StringResourceItem HelpFileTopic { get; set; }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="vbecsBaseForm"></param>
        public PendingTestViewModelBase(VbecsBaseForm vbecsBaseForm, StrRes.StringResourceItem helpFileTopic)
        {
            SetVbecsBaseForm(vbecsBaseForm, false);
            HelpFileTopic = helpFileTopic;

            EnableRejectButton = false;
            EnableAcceptButton = false;

            CommentCommand = new DelegateCommand<ISelectableTestViewModel>(OnCommentCommand);
            CloseCommand = new DelegateCommand(OnCloseCommand);
        }

        /// <summary>
        /// Toggle check box enabled states
        /// BR_115.13 && BR_115.25
        /// </summary>
        /// <param name="selectableModel"></param>
        /// <param name="selectableTestList"></param>
        public void ToggleCheckBoxesEnabledState(ISelectableTestViewModel selectableModel, IEnumerable<ISelectableTestViewModel> selectableTestList)
        {
            bool areAnySelected = selectableTestList.Any(x => x.IsSelected);
            EnableRejectButton = false;
            EnableAcceptButton = false;

            if (areAnySelected)
            {
                if (selectableModel.TestValidation.TestValidationStatus == TestValidationStatus.Valid)
                {
                    EnableRejectButton = true;
                    EnableAcceptButton = true;

                    var notValidList = selectableTestList.Where(x => x.TestValidation.TestValidationStatus != TestValidationStatus.Valid);
                    foreach (var test in notValidList)
                    {
                        test.IsSelectable = false;
                        test.IsSelected = false;
                        test.SelectToolTip = "At least one valid result is selected.";
                    }
                }
                else
                {
                    if (selectableModel.TestValidation.TestValidationStatus == TestValidationStatus.Warning)
                    {
                        EnableRejectButton = true;
                        EnableAcceptButton = true;
                    }
                    else
                    {
                        EnableRejectButton = true;
                    }

                    foreach (var test in selectableTestList)
                    {
                        test.IsSelectable = false;
                        test.IsSelected = false;
                        test.SelectToolTip = "Another non-valid result is selected.";
                    }
                    selectableModel.IsSelectable = true;
                    selectableModel.IsSelected = true;
                    selectableModel.SelectToolTip = null;
                }
            }
            else
            {
                foreach (var test in selectableTestList)
                {
                    test.IsSelectable = true;
                    test.SelectToolTip = null;
                }
            }
        }

        /// <summary>
        /// Add a review comment command
        /// </summary>
        /// <param name="selectableModel"></param>
        private void OnCommentCommand(ISelectableTestViewModel selectableModel)
        {
            if (selectableModel != null)
            {
                DlgOverrideComment dlg = new DlgOverrideComment("Review Comment", "Enter a comment related to this test's review", "AR");
                dlg.SetPicutureImage(DlgOverrideComment.CommentImageType.Comment);
                dlg.ShowDialogFromLockedParent(VbecsBaseForm, false);
                //dlg.ShowDialog(VbecsBaseForm);

                if (dlg.DialogResult == DialogResult.OK)
                {
                    selectableModel.ReviewComment = dlg.Comment;
                }
            }
        }

        /// <summary>
        /// Close the window
        /// </summary>
        private void OnCloseCommand()
        {
            VbecsBaseForm.Close();
        }

        /// <summary>
        /// Uncheck all tests
        /// </summary>
        /// <param name="selectableTestList"></param>
        public void UncheckAll(IEnumerable<ISelectableTestViewModel> selectableTestList)
        {
            foreach (var test in selectableTestList)
            {
                test.IsSelected = false;
                test.IsSelectable = true;
                test.SelectToolTip = null;
            }

            EnableRejectButton = false;
            EnableAcceptButton = false;
        }

        /// <summary>
        /// Equals
        /// </summary>
        /// <param name="other"></param>
        /// <returns></returns>
        public bool Equals(PendingTestViewModelBase other)
        {
            if (ReferenceEquals(null, other)) return false;
            if (ReferenceEquals(this, other)) return true;
            return enableRejectButton.Equals(other.enableRejectButton) && enableAcceptButton.Equals(other.enableAcceptButton);
        }

        /// <summary>
        /// Equals
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            if (ReferenceEquals(null, obj)) return false;
            if (ReferenceEquals(this, obj)) return true;
            if (obj.GetType() != this.GetType()) return false;
            return Equals((PendingTestViewModelBase)obj);
        }

        /// <summary>
        /// GetHashCode
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            unchecked
            {
                return (enableRejectButton.GetHashCode() * 397) ^ enableAcceptButton.GetHashCode();
            }
        }
    }
}
